import * as _algolia_client_common from '@algolia/client-common';
import { CreateClientOptions, RequestOptions, ClientOptions } from '@algolia/client-common';

/**
 * Multiple-testing correction method applied when evaluating metric significance.
 */
type ErrorCorrectionType = 'bonferroni' | 'benjamini-hochberg';

/**
 * Boolean filter applied to the A/B test population. Each filter targets a boolean metric and decides whether to include (true) or exclude (false) matching records.
 */
type MetricsFilter = {
    /**
     * Metric domain (for example `abtesting`, `personalization`).
     */
    domain: string;
    /**
     * Public metric name.
     */
    name: string;
    /**
     * Whether the experiment should record the effects of this filter.
     */
    trackEffects?: boolean | undefined;
    /**
     * If true, keep items that match the filter; if false, exclude them.
     */
    includes?: boolean | undefined;
};

/**
 * Metric for which you want to detect the smallest relative difference.
 */
type EffectMetric = 'addToCartRate' | 'clickThroughRate' | 'conversionRate' | 'purchaseRate' | 'noResultsRate';

/**
 * Configuration for the smallest difference between test variants you want to detect.
 */
type MinimumDetectableEffect = {
    /**
     * Smallest difference in an observable metric between variants. For example, to detect a 10% difference between variants, set this value to 0.1.
     */
    size: number;
    metric: EffectMetric;
};

/**
 * A/B test configuration.
 */
type ABTestConfiguration = {
    minimumDetectableEffect?: MinimumDetectableEffect | undefined;
    /**
     * List of metric filters applied to the test population.
     */
    filters?: Array<MetricsFilter> | undefined;
    errorCorrection?: ErrorCorrectionType | undefined;
};

/**
 * A/B test status.  - `active`. The A/B test is live and search traffic is split between the two variants. - `stopped`. You stopped the A/B test. The A/B test data is still available for analysis. - `expired`. The A/B test was automatically stopped after reaching its end date. - `failed`. Creating the A/B test failed.
 */
type Status = 'active' | 'stopped' | 'expired' | 'failed';

/**
 * Metric specific metadata.
 */
type MetricMetadata = {
    /**
     * Only present in case the metric is \'revenue\'. It is the amount exceeding the 95th percentile of global revenue transactions involved in the AB Test. This amount is not considered when calculating statistical significance. It is tied to a per revenue-currency pair contrary to other global filter effects (such as outliers and empty search count).
     */
    winsorizedValue?: number | undefined;
    /**
     * Mean value for this metric.
     */
    mean?: number | undefined;
};

type MetricResult = {
    name: string;
    /**
     * Date and time when the metric was last updated, in RFC 3339 format.
     */
    updatedAt: string;
    value: number;
    /**
     * The upper bound of the 95% confidence interval for the metric value. The confidence interval is calculated using either the relative ratio or relative difference between the metric values for the control and the variant. Relative ratio is used for metrics that are ratios (e.g., click-through rate, conversion rate), while relative difference is used for continuous metrics (e.g., revenue).
     */
    valueCIHigh?: number | undefined;
    /**
     * The lower bound of the 95% confidence interval for the metric value. The confidence interval is calculated using either the relative ratio or relative difference between the metric values for the control and the variant. Relative ratio is used for metrics that are ratios (e.g., click-through rate, conversion rate), while relative difference is used for continuous metrics (e.g., revenue).
     */
    valueCILow?: number | undefined;
    /**
     * PValue for the first variant (control) will always be 0. For the other variants, pValue is calculated for the current variant based on the control.
     */
    pValue: number;
    /**
     * Dimension defined during test creation.
     */
    dimension?: string | undefined;
    metadata?: MetricMetadata | undefined;
    /**
     * The value that was computed during error correction. It is used to determine significance of the metric pValue. The critical value is calculated using Bonferroni or Benjamini-Hochberg corrections, based on the given configuration during the A/B test creation.
     */
    criticalValue?: number | undefined;
    /**
     * Whether the pValue is significant or not based on the critical value and the error correction algorithm used.
     */
    significant?: boolean | undefined;
};

/**
 * Empty searches removed from the A/B test as a result of configuration settings.
 */
type EmptySearchFilter = {
    /**
     * Number of users removed from the A/B test.
     */
    usersCount?: number | undefined;
    /**
     * Number of tracked searches removed from the A/B test.
     */
    trackedSearchesCount?: number | undefined;
};

/**
 * Outliers removed from the A/B test as a result of configuration settings.
 */
type OutliersFilter = {
    /**
     * Number of users removed from the A/B test.
     */
    usersCount?: number | undefined;
    /**
     * Number of tracked searches removed from the A/B test.
     */
    trackedSearchesCount?: number | undefined;
};

/**
 * A/B test filter effects resulting from configuration settings.
 */
type FilterEffects = {
    outliers?: OutliersFilter | undefined;
    emptySearch?: EmptySearchFilter | undefined;
};

/**
 * Variant specific metadata.
 */
type VariantMetadata = {
    filterEffects?: FilterEffects | undefined;
};

type Variant = {
    /**
     * Description for this variant.
     */
    description: string;
    /**
     * Estimated number of searches required to achieve the desired statistical significance.  The A/B test configuration must include a `minimumDetectableEffect` setting for this number to be included in the response.
     */
    estimatedSampleSize?: number | undefined;
    /**
     * Index name of the A/B test variant (case-sensitive).
     */
    index: string;
    /**
     * Percentage of search requests each variant receives.
     */
    trafficPercentage: number;
    /**
     * All ABTest metrics that were defined during test creation.
     */
    metrics: Array<MetricResult>;
    metadata?: VariantMetadata | undefined;
    /**
     * Search parameters applied to this variant when the same index is used for multiple variants. Only present if custom search parameters were provided during test creation.
     */
    customSearchParameters?: Record<string, unknown> | undefined;
};

type ABTest = {
    /**
     * Unique A/B test identifier.
     */
    abTestID: number;
    /**
     * Date and time when the A/B test was last updated, in RFC 3339 format.
     */
    updatedAt: string;
    /**
     * Date and time when the A/B test was created, in RFC 3339 format.
     */
    createdAt: string;
    /**
     * End date and time of the A/B test, in RFC 3339 format.
     */
    endAt: string;
    /**
     * A/B test name.
     */
    name: string;
    status: Status;
    /**
     * A/B test variants.  The first variant is your _control_ index, typically your production index. All of the additional variants are indexes with changed settings that you want to test against the control.
     */
    variants: Array<Variant>;
    configuration?: ABTestConfiguration | undefined;
    /**
     * Unique migrated A/B test identifier.
     */
    migratedAbTestID?: number | undefined;
};

type ABTestResponse = {
    /**
     * Index name of the A/B test variant (case-sensitive).
     */
    index: string;
    /**
     * Unique A/B test identifier.
     */
    abTestID: number;
    /**
     * Unique identifier of a task.  A successful API response means that a task was added to a queue. It might not run immediately. You can check the task\'s progress with the [`task` operation](#tag/Indices/operation/getTask) and this `taskID`.
     */
    taskID: number;
};

type AbTestsVariant = {
    /**
     * Index name of the A/B test variant (case-sensitive).
     */
    index: string;
    /**
     * Percentage of search requests each variant receives.
     */
    trafficPercentage: number;
    /**
     * Description for this variant.
     */
    description?: string | undefined;
};

/**
 * Search parameters to add to the test variant. Only use this parameter if the two variants use the same index.
 */
type CustomSearchParams = {
    customSearchParameters: Record<string, unknown>;
};

type AbTestsVariantSearchParams = AbTestsVariant & CustomSearchParams;

type AddABTestsVariant = AbTestsVariant | AbTestsVariantSearchParams;

/**
 * Defines a metric to be retrieved during an A/B test.
 */
type CreateMetric = {
    /**
     * Name of the metric.
     */
    name: string;
    /**
     * Dimension of the metric, for example, in case of a revenue metric it could be USD, EUR...
     */
    dimension?: string | undefined;
};

type AddABTestsRequest = {
    /**
     * A/B test name.
     */
    name: string;
    /**
     * A/B test variants.
     */
    variants: Array<AddABTestsVariant>;
    /**
     * A/B test metrics involved in the test. Only these metrics will be considered when calculating results.
     */
    metrics: Array<CreateMetric>;
    configuration?: ABTestConfiguration | undefined;
    /**
     * End date and time of the A/B test, in RFC 3339 format.
     */
    endAt: string;
};

/**
 * A/B test configuration for estimating the sample size and duration using minimum detectable effect.
 */
type EstimateConfiguration = {
    /**
     * List of metric filters applied to the test population.
     */
    filters?: Array<MetricsFilter> | undefined;
    minimumDetectableEffect: MinimumDetectableEffect;
};

type EstimateABTestRequest = {
    configuration: EstimateConfiguration;
    /**
     * A/B test variants.
     */
    variants: Array<AddABTestsVariant>;
};

type EstimateABTestResponse = {
    /**
     * Estimated number of days needed to reach the sample sizes required for detecting the configured effect. This value is based on historical traffic.
     */
    durationDays?: number | undefined;
    /**
     * Sample size estimates for each variant. The first element is the control variant. Each element is the estimated number of searches required to achieve the desired statistical significance.
     */
    sampleSizes?: Array<number> | undefined;
};

type ListABTestsResponse = {
    /**
     * A/B tests.
     */
    abtests: Array<ABTest> | null;
    /**
     * Number of A/B tests.
     */
    count: number;
    /**
     * Number of retrievable A/B tests.
     */
    total: number;
};

type ScheduleABTestResponse = {
    /**
     * Unique scheduled A/B test identifier.
     */
    abTestScheduleID: number;
};

type ScheduleABTestsRequest = {
    /**
     * A/B test name.
     */
    name: string;
    /**
     * A/B test variants.
     */
    variants: Array<AddABTestsVariant>;
    /**
     * A/B test metrics involved in the test. Only these metrics will be considered when calculating results.
     */
    metrics: Array<CreateMetric>;
    configuration?: ABTestConfiguration | undefined;
    /**
     * Date and time when the A/B test is scheduled to start, in RFC 3339 format.
     */
    scheduledAt: string;
    /**
     * End date and time of the A/B test, in RFC 3339 format.
     */
    endAt: string;
};

type MetricDate = {
    /**
     * Date where the metric was updated, in RFC 3339 format.
     */
    date?: string | undefined;
    /**
     * All ABTest metrics that were defined during test creation.
     */
    metrics?: Array<MetricResult> | undefined;
};

type TimeseriesVariant = {
    dates?: Array<MetricDate> | undefined;
};

type Timeseries = {
    /**
     * Unique A/B test identifier.
     */
    abTestID: number;
    /**
     * A/B test timeseries variants.  The first variant is your _control_ index, typically your production index. All of the additional variants are indexes with changed settings that you want to test against the control.
     */
    variants: Array<TimeseriesVariant>;
};

/**
 * Sort order for A/B tests by start date. Use \'asc\' for ascending or \'desc\' for descending. Active A/B tests are always listed first.
 */
type Direction = 'asc' | 'desc';

type MetricName = 'search_count' | 'tracked_search_count' | 'user_count' | 'tracked_user_count' | 'no_result_count' | 'add_to_cart_count' | 'purchase_count' | 'clicked_search_count' | 'converted_search_count' | 'click_through_rate' | 'conversion_rate' | 'add_to_cart_rate' | 'purchase_rate' | 'average_click_position' | 'revenue';

/**
 * Properties for the `customDelete` method.
 */
type CustomDeleteProps = {
    /**
     * Path of the endpoint, for example `1/newFeature`.
     */
    path: string;
    /**
     * Query parameters to apply to the current query.
     */
    parameters?: {
        [key: string]: any;
    } | undefined;
};
/**
 * Properties for the `customGet` method.
 */
type CustomGetProps = {
    /**
     * Path of the endpoint, for example `1/newFeature`.
     */
    path: string;
    /**
     * Query parameters to apply to the current query.
     */
    parameters?: {
        [key: string]: any;
    } | undefined;
};
/**
 * Properties for the `customPost` method.
 */
type CustomPostProps = {
    /**
     * Path of the endpoint, for example `1/newFeature`.
     */
    path: string;
    /**
     * Query parameters to apply to the current query.
     */
    parameters?: {
        [key: string]: any;
    } | undefined;
    /**
     * Parameters to send with the custom request.
     */
    body?: Record<string, unknown> | undefined;
};
/**
 * Properties for the `customPut` method.
 */
type CustomPutProps = {
    /**
     * Path of the endpoint, for example `1/newFeature`.
     */
    path: string;
    /**
     * Query parameters to apply to the current query.
     */
    parameters?: {
        [key: string]: any;
    } | undefined;
    /**
     * Parameters to send with the custom request.
     */
    body?: Record<string, unknown> | undefined;
};
/**
 * Properties for the `deleteABTest` method.
 */
type DeleteABTestProps = {
    /**
     * Unique A/B test identifier.
     */
    id: number;
};
/**
 * Properties for the `getABTest` method.
 */
type GetABTestProps = {
    /**
     * Unique A/B test identifier.
     */
    id: number;
};
/**
 * Properties for the `getTimeseries` method.
 */
type GetTimeseriesProps = {
    /**
     * Unique A/B test identifier.
     */
    id: number;
    /**
     * Start date of the period to analyze, in `YYYY-MM-DD` format.
     */
    startDate?: string | undefined;
    /**
     * End date of the period to analyze, in `YYYY-MM-DD` format.
     */
    endDate?: string | undefined;
    /**
     * List of metrics to retrieve. If not specified, all metrics are returned.
     */
    metric?: Array<MetricName> | undefined;
};
/**
 * Properties for the `listABTests` method.
 */
type ListABTestsProps = {
    /**
     * Position of the first item to return.
     */
    offset?: number | undefined;
    /**
     * Number of items to return.
     */
    limit?: number | undefined;
    /**
     * Index name prefix. Only A/B tests for indices starting with this string are included in the response.
     */
    indexPrefix?: string | undefined;
    /**
     * Index name suffix. Only A/B tests for indices ending with this string are included in the response.
     */
    indexSuffix?: string | undefined;
    /**
     * Sort order for A/B tests by start date. Use \'asc\' for ascending or \'desc\' for descending. Active A/B tests are always listed first.
     */
    direction?: Direction | undefined;
};
/**
 * Properties for the `stopABTest` method.
 */
type StopABTestProps = {
    /**
     * Unique A/B test identifier.
     */
    id: number;
};

declare const apiClientVersion = "1.1.0";
declare const REGIONS: readonly ["de", "us"];
type Region = (typeof REGIONS)[number];
type RegionOptions = {
    region?: Region | undefined;
};
declare function createAbtestingV3Client({ appId: appIdOption, apiKey: apiKeyOption, authMode, algoliaAgents, region: regionOption, ...options }: CreateClientOptions & RegionOptions): {
    transporter: _algolia_client_common.Transporter;
    /**
     * The `appId` currently in use.
     */
    appId: string;
    /**
     * The `apiKey` currently in use.
     */
    apiKey: string;
    /**
     * Clears the cache of the transporter for the `requestsCache` and `responsesCache` properties.
     */
    clearCache(): Promise<void>;
    /**
     * Get the value of the `algoliaAgent`, used by our libraries internally and telemetry system.
     */
    readonly _ua: string;
    /**
     * Adds a `segment` to the `x-algolia-agent` sent with every requests.
     *
     * @param segment - The algolia agent (user-agent) segment to add.
     * @param version - The version of the agent.
     */
    addAlgoliaAgent(segment: string, version?: string | undefined): void;
    /**
     * Helper method to switch the API key used to authenticate the requests.
     *
     * @param params - Method params.
     * @param params.apiKey - The new API Key to use.
     */
    setClientApiKey({ apiKey }: {
        apiKey: string;
    }): void;
    /**
     * Creates a new A/B test.
     *
     * Required API Key ACLs:
     *  - editSettings
     * @param addABTestsRequest - The addABTestsRequest object.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    addABTests(addABTestsRequest: AddABTestsRequest, requestOptions?: RequestOptions): Promise<ABTestResponse>;
    /**
     * This method lets you send requests to the Algolia REST API.
     * @param customDelete - The customDelete object.
     * @param customDelete.path - Path of the endpoint, for example `1/newFeature`.
     * @param customDelete.parameters - Query parameters to apply to the current query.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    customDelete({ path, parameters }: CustomDeleteProps, requestOptions?: RequestOptions): Promise<Record<string, unknown>>;
    /**
     * This method lets you send requests to the Algolia REST API.
     * @param customGet - The customGet object.
     * @param customGet.path - Path of the endpoint, for example `1/newFeature`.
     * @param customGet.parameters - Query parameters to apply to the current query.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    customGet({ path, parameters }: CustomGetProps, requestOptions?: RequestOptions): Promise<Record<string, unknown>>;
    /**
     * This method lets you send requests to the Algolia REST API.
     * @param customPost - The customPost object.
     * @param customPost.path - Path of the endpoint, for example `1/newFeature`.
     * @param customPost.parameters - Query parameters to apply to the current query.
     * @param customPost.body - Parameters to send with the custom request.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    customPost({ path, parameters, body }: CustomPostProps, requestOptions?: RequestOptions): Promise<Record<string, unknown>>;
    /**
     * This method lets you send requests to the Algolia REST API.
     * @param customPut - The customPut object.
     * @param customPut.path - Path of the endpoint, for example `1/newFeature`.
     * @param customPut.parameters - Query parameters to apply to the current query.
     * @param customPut.body - Parameters to send with the custom request.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    customPut({ path, parameters, body }: CustomPutProps, requestOptions?: RequestOptions): Promise<Record<string, unknown>>;
    /**
     * Deletes an A/B test by its ID.
     *
     * Required API Key ACLs:
     *  - editSettings
     * @param deleteABTest - The deleteABTest object.
     * @param deleteABTest.id - Unique A/B test identifier.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    deleteABTest({ id }: DeleteABTestProps, requestOptions?: RequestOptions): Promise<ABTestResponse>;
    /**
     * Given the traffic percentage and the expected effect size, this endpoint estimates the sample size and duration of an A/B test based on historical traffic.
     *
     * Required API Key ACLs:
     *  - analytics
     * @param estimateABTestRequest - The estimateABTestRequest object.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    estimateABTest(estimateABTestRequest: EstimateABTestRequest, requestOptions?: RequestOptions): Promise<EstimateABTestResponse>;
    /**
     * Retrieves the details for an A/B test by its ID.
     *
     * Required API Key ACLs:
     *  - analytics
     * @param getABTest - The getABTest object.
     * @param getABTest.id - Unique A/B test identifier.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    getABTest({ id }: GetABTestProps, requestOptions?: RequestOptions): Promise<ABTest>;
    /**
     * Retrieves timeseries for an A/B test by its ID.
     *
     * Required API Key ACLs:
     *  - analytics
     * @param getTimeseries - The getTimeseries object.
     * @param getTimeseries.id - Unique A/B test identifier.
     * @param getTimeseries.startDate - Start date of the period to analyze, in `YYYY-MM-DD` format.
     * @param getTimeseries.endDate - End date of the period to analyze, in `YYYY-MM-DD` format.
     * @param getTimeseries.metric - List of metrics to retrieve. If not specified, all metrics are returned.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    getTimeseries({ id, startDate, endDate, metric }: GetTimeseriesProps, requestOptions?: RequestOptions): Promise<Timeseries>;
    /**
     * Lists all A/B tests you configured for this application.
     *
     * Required API Key ACLs:
     *  - analytics
     * @param listABTests - The listABTests object.
     * @param listABTests.offset - Position of the first item to return.
     * @param listABTests.limit - Number of items to return.
     * @param listABTests.indexPrefix - Index name prefix. Only A/B tests for indices starting with this string are included in the response.
     * @param listABTests.indexSuffix - Index name suffix. Only A/B tests for indices ending with this string are included in the response.
     * @param listABTests.direction - Sort order for A/B tests by start date. Use \'asc\' for ascending or \'desc\' for descending. Active A/B tests are always listed first.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    listABTests({ offset, limit, indexPrefix, indexSuffix, direction }?: ListABTestsProps, requestOptions?: RequestOptions | undefined): Promise<ListABTestsResponse>;
    /**
     * Schedule an A/B test to be started at a later time.
     *
     * Required API Key ACLs:
     *  - editSettings
     * @param scheduleABTestsRequest - The scheduleABTestsRequest object.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    scheduleABTest(scheduleABTestsRequest: ScheduleABTestsRequest, requestOptions?: RequestOptions): Promise<ScheduleABTestResponse>;
    /**
     * Stops an A/B test by its ID.  You can\'t restart stopped A/B tests.
     *
     * Required API Key ACLs:
     *  - editSettings
     * @param stopABTest - The stopABTest object.
     * @param stopABTest.id - Unique A/B test identifier.
     * @param requestOptions - The requestOptions to send along with the query, they will be merged with the transporter requestOptions.
     */
    stopABTest({ id }: StopABTestProps, requestOptions?: RequestOptions): Promise<ABTestResponse>;
};

/**
 * Error.
 */
type ErrorBase = Record<string, any> & {
    message?: string | undefined;
};

type AbtestingV3Client = ReturnType<typeof createAbtestingV3Client>;

declare function abtestingV3Client(appId: string, apiKey: string, region?: Region | undefined, options?: ClientOptions | undefined): AbtestingV3Client;

export { type ABTest, type ABTestConfiguration, type ABTestResponse, type AbTestsVariant, type AbTestsVariantSearchParams, type AbtestingV3Client, type AddABTestsRequest, type AddABTestsVariant, type CreateMetric, type CustomDeleteProps, type CustomGetProps, type CustomPostProps, type CustomPutProps, type CustomSearchParams, type DeleteABTestProps, type Direction, type EffectMetric, type EmptySearchFilter, type ErrorBase, type ErrorCorrectionType, type EstimateABTestRequest, type EstimateABTestResponse, type EstimateConfiguration, type FilterEffects, type GetABTestProps, type GetTimeseriesProps, type ListABTestsProps, type ListABTestsResponse, type MetricDate, type MetricMetadata, type MetricName, type MetricResult, type MetricsFilter, type MinimumDetectableEffect, type OutliersFilter, type Region, type RegionOptions, type ScheduleABTestResponse, type ScheduleABTestsRequest, type Status, type StopABTestProps, type Timeseries, type TimeseriesVariant, type Variant, type VariantMetadata, abtestingV3Client, apiClientVersion };
